/*
    SPDX-FileCopyrightText: 2013 Marco Martin <mart@kde.org>
    SPDX-FileCopyrightText: 2014 Sebastian Kügler <sebas@kde.org>

    SPDX-License-Identifier: GPL-2.0-or-later
*/

import QtQuick
import QtQuick.Controls as QtControls2
import Qt5Compat.GraphicalEffects

import org.kde.kirigami as Kirigami
import org.kde.kquickcontrolsaddons
import org.kde.kcmutils as KCM

KCM.GridDelegate {
  id: wallpaperDelegate

  property alias color: backgroundRect.color
  property alias previewSize: previewImage.sourceSize
  opacity: model.pendingDeletion ? 0.5 : 1
  scale: index, 1 // Workaround for https://bugreports.qt.io/browse/QTBUG-107458

  text: model.display
  subtitle: model.author

  hoverEnabled: true

  actions: [
    Kirigami.Action {
      icon.name: "document-open-folder"
      tooltip: i18nd("plasma_wallpaper_org.kde.image", "Open Containing Folder")
      onTriggered: imageModel.openContainingFolder(index)
    },
    Kirigami.Action {
      icon.name: "edit-undo"
      visible: model.pendingDeletion
      tooltip: i18nd("plasma_wallpaper_org.kde.image", "Restore wallpaper")
      onTriggered: model.pendingDeletion = false
    },
    Kirigami.Action {
      icon.name: "edit-delete"
      tooltip: i18nd("plasma_wallpaper_org.kde.image", "Remove Wallpaper")
      visible: model.removable && !model.pendingDeletion && !cfg_IsSlideshow
      onTriggered: {
        model.pendingDeletion = true;
        if (wallpapersGrid.view.currentIndex === index) {
          const newIndex = (index + 1) % (imageModel.count - 1);
          wallpapersGrid.view.itemAtIndex(newIndex).clicked();
        }
        root.configurationChanged(); // BUG 438585
      }
    }
  ]

thumbnail: Rectangle {
  id: backgroundRect
  color: cfg_Color
  anchors.fill: parent

  Kirigami.Icon {
    anchors.centerIn: parent
    width: Kirigami.Units.iconSizes.large
    height: width
    source: "view-preview"
    visible: previewImage.status != Image.Ready
  }

  FastBlur {
    id: fastBlur
    visible: cfg_Blur
    anchors.fill: parent
    radius: 4
    source: Image {
      asynchronous: true
      cache: false
      fillMode: Image.PreserveAspectCrop
      source: fastBlur.visible ? previewImage.source : ""
      sourceSize: previewImage.sourceSize
      visible: false
    }
  }

  Image {
    id: previewImage
    anchors.fill: parent
    asynchronous: true
    cache: false
    fillMode: cfg_FillMode
    source: model.preview

    layer.enabled: cfg_ActiveBlur

    layer.effect: FastBlur {
      visible: cfg_ActiveBlur
      anchors.fill: parent
      radius: wallpaperDelegate.hovered ? cfg_BlurRadius : 0
      source: Image {
        asynchronous: true
        cache: false
        fillMode: Image.PreserveAspectCrop
        source: previewImage.source
        sourceSize: previewImage.sourceSize
        anchors.fill: parent
      }
      // animate the blur apparition
      Behavior on radius {
        NumberAnimation {
          duration: cfg_AnimationDuration
        }
      }
    }
  }

  ColorOverlay {
    id: activeColorOverlay
    anchors.fill: previewImage
    source: previewImage
    visible: cfg_ActiveColor
    color: cfg_ActiveColorColor
    opacity: (cfg_ActiveColor && wallpaperDelegate.hovered) ? cfg_ActiveColorTransparency / 100 : 0

    // animate the color apparition
    Behavior on opacity {
      NumberAnimation {
        duration: cfg_AnimationDuration
      }
    }
  }

  QtControls2.CheckBox {
    visible: cfg_IsSlideshow
    anchors.right: parent.right
    anchors.top: parent.top
    checked: visible ? model.checked : false
    onToggled: model.checked = checked
  }

  Behavior on color {
    ColorAnimation {
      duration: Kirigami.Units.longDuration
      easing.type: Easing.InOutQuad
    }
  }
}

Behavior on opacity {
  OpacityAnimator {
    duration: Kirigami.Units.longDuration
    easing.type: Easing.InOutQuad
  }
}

onClicked: {
  if (!cfg_IsSlideshow) {
    cfg_Image = model.packageName || model.path;
    if (typeof wallpaper !== "undefined") {
      wallpaper.configuration.PreviewImage = cfg_Image;
    }
  }
  GridView.currentIndex = index;
}
}
