/*
    SPDX-FileCopyrightText: 2025 Alexandru Balan
    pearOS lockscreen theme for KDE Plasma

    SPDX-License-Identifier: PPLv2
*/

import QtQuick
import QtQuick.Layouts
import QtQuick.Controls
import Qt5Compat.GraphicalEffects
import org.kde.kirigami 2.20 as Kirigami
import org.kde.kscreenlocker 1.0 as ScreenLocker
import org.kde.plasma.private.sessions 2.0

Item {
    id: root

    // Properties that kscreenlocker expects
    property bool locked: false
    property bool viewVisible: false
    property bool suspendToRamSupported: false
    property bool suspendToDiskSupported: false
    property string notification: ""
    signal clearPassword()
    signal notificationRepeated()
    signal suspendToRam()
    signal suspendToDisk()

    LayoutMirroring.enabled: Qt.application.layoutDirection === Qt.RightToLeft
    LayoutMirroring.childrenInherit: true

    implicitWidth: 800
    implicitHeight: 600

    readonly property bool softwareRendering: GraphicsInfo.api === GraphicsInfo.Software
    property int sizeAvatar: 80

    FontLoader {
        id: fontbold
        source: Qt.resolvedUrl("../../fonts/SFUIText-Semibold.otf")
    }

    SessionManagement {
        id: sessionManagement
    }

    Connections {
        target: authenticator
        function onFailed() {
            password.placeholderText = i18nd("kscreenlocker_greet", "Unlocking failed")
            password.text = ""
            errorMsgContainer.visible = true
        }
        function onSucceeded() {
            Qt.quit()
        }
        function onPromptForSecretChanged() {
            if (authenticator.promptForSecret)
                authenticator.respond(password.text)
        }
    }

    Binding {
        target: PasswordSync
        property: "password"
        value: password.text
    }

    // Fundal /usr/share/extras/ (acoperă wallpaper-ul din setări)
    Rectangle {
        id: backgroundFallback
        anchors.fill: parent
        color: "#1a1a2e"
        z: -1
    }
    Image {
        id: backgroundImage
        anchors.fill: parent
        fillMode: Image.PreserveAspectCrop
        z: -1
        source: "file:///usr/share/extras/background.jpg"
        onStatusChanged: {
            if (status === Image.Error) {
                if (source === "file:///usr/share/extras/background.jpg")
                    source = "file:///usr/share/extras/background.png"
                else
                    visible = false
            }
        }
    }

    // Overlay întunecat pentru contrast
    Rectangle {
        id: backgroundOverlay
        anchors.fill: parent
        color: "#40000000"
    }

    // Clock area – la fel ca default (ceas sus, inline)
    Item {
        id: identclock
        width: parent.width
        height: parent.height
        opacity: 0.8

        property date currentTime: new Date()
        Timer {
            interval: 1000
            running: true
            repeat: true
            onTriggered: identclock.currentTime = new Date()
        }

        ColumnLayout {
            anchors.horizontalCenter: parent.horizontalCenter
            anchors.top: parent.top
            anchors.topMargin: 100
            spacing: 2
            Label {
                text: Qt.formatDateTime(identclock.currentTime, "ddd MMM d")
                color: "white"
                opacity: 0.5
                font.pointSize: 20
                font.weight: Font.DemiBold
                font.capitalization: Font.Capitalize
                Layout.alignment: Qt.AlignHCenter
                font.family: fontbold.name
            }
            Label {
                text: Qt.formatDateTime(identclock.currentTime, "hh:mm")
                color: "white"
                opacity: 0.5
                font.pointSize: 100
                font.bold: true
                Layout.alignment: Qt.AlignHCenter
                font.family: fontbold.name
            }
        }
    }

    // User dialog area – la fel ca default (avatarToPasswordGap 28)
    Rectangle {
        id: baseOfUserDialog
        readonly property real avatarToPasswordGap: 28
        property real contentAreaHeight: sizeAvatar * 0.9 + avatarToPasswordGap + 32 + 12 + 14 + 4 + 10
        width: Math.max(150, sizeAvatar * 0.9 + 40)
        height: contentAreaHeight + 15
        anchors.horizontalCenter: parent.horizontalCenter
        anchors.bottom: parent.bottom
        anchors.bottomMargin: 40
        color: "transparent"
        focus: true

        Item {
            id: avatarContainer
            width: sizeAvatar * 0.9
            height: sizeAvatar * 0.9
            anchors.top: parent.top
            anchors.horizontalCenter: parent.horizontalCenter

            Image {
                id: avaImg
                width: parent.width * 1.5
                height: parent.height * 1.5
                anchors.centerIn: parent
                fillMode: Image.PreserveAspectCrop
                visible: false
                source: (typeof kscreenlocker_userImage !== "undefined" && kscreenlocker_userImage !== "") ? "file://" + kscreenlocker_userImage.split("/").map(encodeURIComponent).join("/") : Qt.resolvedUrl("../../faces/.face.icon")
            }

            Rectangle {
                id: avatarMask
                width: avatarContainer.width
                height: avatarContainer.height
                radius: width / 2
                visible: false
            }

            OpacityMask {
                anchors.fill: avatarMask
                source: avaImg
                maskSource: avatarMask
            }
        }

        Text {
            id: usernametext
            text: kscreenlocker_userName.length > 0 ? kscreenlocker_userName : i18nd("kscreenlocker_greet", "The session is locked")
            anchors.top: parent.top
            anchors.topMargin: sizeAvatar * 0.9 + 6
            anchors.horizontalCenter: parent.horizontalCenter
            font.pixelSize: 14
            font.family: fontbold.name
            font.capitalization: Font.MixedCase
            font.bold: true
            color: "white"
        }

        TextField {
            id: password
            anchors.top: parent.top
            anchors.topMargin: sizeAvatar * 0.9 + baseOfUserDialog.avatarToPasswordGap
            anchors.horizontalCenter: parent.horizontalCenter
            height: 32
            width: 220
            color: "#fff"
            placeholderTextColor: "#66FFFFFF"
            placeholderText: i18nd("kscreenlocker_greet", "Enter Password")
            echoMode: TextInput.Password
            focus: true
            font.weight: Font.DemiBold
            font.family: fontbold.name
            horizontalAlignment: TextInput.AlignHCenter
            verticalAlignment: TextInput.AlignVCenter
            enabled: !authenticator.busy

            onAccepted: authenticator.startAuthenticating(password.text)

            background: Rectangle {
                implicitWidth: parent.width
                implicitHeight: parent.height
                color: "#fff"
                opacity: 0.2
                radius: 15
            }
        }

        Rectangle {
            id: errorMsgContainer
            visible: false
            color: "transparent"
            anchors.top: password.bottom
            anchors.horizontalCenter: parent.horizontalCenter
        }

        Text {
            id: touchIdLabel
            text: i18nd("kscreenlocker_greet", "Touch ID or Enter Password (plasma shell)")
            color: "#fff"
            font.pointSize: 8
            anchors.top: password.bottom
            anchors.topMargin: 12
            anchors.horizontalCenter: parent.horizontalCenter
        }

        Text {
            id: greetingLabel
            text: i18nd("kscreenlocker_greet", "(or place your fingerprint on the reader)")
            color: "#fff"
            visible: authenticator.authenticatorTypes & ScreenLocker.Authenticator.Fingerprint
            font.pointSize: 8
            anchors.top: touchIdLabel.bottom
            anchors.topMargin: 4
            anchors.horizontalCenter: parent.horizontalCenter
        }
    }

    // Power buttons - top right
    Row {
        anchors.top: parent.top
        anchors.right: parent.right
        anchors.rightMargin: 15
        anchors.topMargin: 15
        spacing: 20

        Image {
            id: suspendBtn
            height: 22
            width: 22
            visible: root.suspendToRamSupported
            source: Qt.resolvedUrl("../../images/system-suspend.svg")
            fillMode: Image.PreserveAspectFit
            MouseArea {
                anchors.fill: parent
                hoverEnabled: true
                onEntered: suspendBtn.source = Qt.resolvedUrl("../../images/system-suspend-hover.svg")
                onExited: suspendBtn.source = Qt.resolvedUrl("../../images/system-suspend.svg")
                onClicked: root.suspendToRam()
            }
        }

        Image {
            id: switchUserBtn
            height: 22
            width: 22
            visible: sessionManagement.canSwitchUser
            source: Qt.resolvedUrl("../../images/go-down.svg")
            fillMode: Image.PreserveAspectFit
            MouseArea {
                anchors.fill: parent
                hoverEnabled: true
                onClicked: sessionManagement.switchUser()
            }
        }
    }

    function resetPasswordField() {
        password.text = ""
        password.placeholderText = i18nd("kscreenlocker_greet", "Enter Password")
        errorMsgContainer.visible = false
        clearPassword()
    }

    Keys.onPressed: function(event) {
        if (event.key === Qt.Key_Return || event.key === Qt.Key_Enter) {
            authenticator.startAuthenticating(password.text)
            event.accepted = true
        } else if (event.key === Qt.Key_Escape) {
            resetPasswordField()
        }
    }

    Component.onCompleted: {
        password.forceActiveFocus()
    }
}
